// IFDL Main Procedure
//  1990, 1994, 1998 WaveMetrics, Inc. all rights reserved
// The full (purchased) IFDL requires an alias/shortcut to the IFDL extension
// (or the IFDL noFPU extension) to be in the Igor Extensions folder.
//
// The demo version of IFDL does not, but the demo version can design only window-based
// filters (Kaiser Window and Window Filters)
//
// To use, type:
// #include ":IFDL Procedures:IFDL"
// in your procedure window.

#pragma version=3.1
#pragma rtGlobals=1		// Use modern global access method.

// Other IFDL procedures
#include ":IFDL Procedures:MPR Low Pass"
#include ":IFDL Procedures:MPR 2 Band"
#include ":IFDL Procedures:MPR 2 Band Notch"
#include ":IFDL Procedures:MPR 3 Band"
#include ":IFDL Procedures:MPR Differentiator"
#include ":IFDL Procedures:MPR Hilbert"
#include ":IFDL Procedures:MPR Arbitrary"
#include ":IFDL Procedures:Kaiser Low Pass"
#include ":IFDL Procedures:Kaiser Maximally Flat"
#include ":IFDL Procedures:Window Filters"
#include ":IFDL Procedures:Compare Filters"

// Support Procedures
#include <Strings as Lists>
#include <Keyword-Value>
#include <Readback ModifyStr>
#include <Axis Utilities>
#include <Math Utility Functions>

Menu "IFDL"
	"Set IFDL Parameters...", InitializeIFDL()
	"-"
	"FIR Filter Designs", DesignPanel()
	"Normalize Design Coefficients..."
	"-"
	Submenu "Save Design"
		"Descriptive Text File..."
		"C Include File..."
		"Igor Binary File..."
		"As Another Wave..."
	End
	"Design Report Layout", CreateReport()
	"-"
	"Compare 2 Filters", Compare2Filters("compare")
	"Sine Sweep Test Signal..."
	"Apply Filter..."
	"-"
	"Close All IFDL Windows"
	"Delete All IFDL Data"
End

Function IsIFDLDemo()
	return exists("Remez") != 4	// full IFDL requires IFDL XOP
End

Function IsIFDLInitialized()
	return NumVarOrDefault("root:Packages:WM_IFDL:fs",-1) > 0 
End

// returns 1 if IFDL initialization was cancelled
// returns 0 if IFDL was initialized successfully, or none was needed
Proc GuaranteeIFDLInitialized()
	if( IsIFDLInitialized() )
		return 0
	endif
	InitializeIFDL()	// InitializeIFDL() calls DesignPanel() if it doesn't exist
End

Proc InitializeIFDL()
	String dfSav= Set_IFDL_DataFolder()
	Variable/G saidIt
	if( IsIFDLDemo() )
		if (! saidIt )
			DoAlert 0, "Running in demo mode. Purchase IFDL and install the proper IFDL extension into the Igor Extensions folder to enable MPR and Kaiser MaxFlat designs."
			saidIt= 1
		endif
	endif
	SetDataFolder dfSav
	SetIFDLParameters()
End

Proc ResetIFDL()
	String dfSav= Set_IFDL_DataFolder()
	// delete the window coordinates, reset saidIt
	KillWaves/Z W_WindowCoordinates 
	KillVariables/Z  saidIt
	SetDataFolder dfSav
End

Proc SetIFDLParameters(fwv,fSamp,fftPointsStr,qMode,qBits,csl,afss)
	String fwv=StrVarOrDefault("root:Packages:WM_IFDL:freqWave","_value entered manually_")
	Variable fSamp=NumVarOrDefault("root:Packages:WM_IFDL:fs",1)
	String fftPointsStr=num2istr(NumVarOrDefault("root:Packages:WM_IFDL:fftLen",256))	// freq-domain values (actually one more is used)
	Variable qMode=NumVarOrDefault("root:Packages:WM_IFDL:quanType",1)
	Variable qBits=NumVarOrDefault("root:Packages:WM_IFDL:quanBits",53)
	Variable csl=NumVarOrDefault("root:Packages:WM_IFDL:ndlyCsl",1)
	String afss=StrVarOrDefault("root:Packages:WM_IFDL:annoFontSizeStr","auto")
	Prompt fwv "get sampling frequency from:",popup,"_value entered manually_;"+WaveList("*",";","")
	Prompt fSamp "manual sampling frequency, Hz"
	Prompt fftPointsStr "frequency resolution (# of FFT values)",popup,"128;256;512;1024;2048;4096;8192;16384;32768;65536"
	Prompt qMode "quantization type",popup,";floating point;fixed point"
	Prompt qBits "bits of quantization"
	Prompt csl "filtered output timing",popup,"in phase with input;delayed (causal)"
	Prompt afss "font size for IFDL legends (points)",popup,"auto;5;6;7;8;9;10;11;12;13;14;15;16;18;20;24;28;32;36;40;44;48;56;72;"

	PauseUpdate;Silent 1
	// If the wave exists, set frequency from selected wave
	if( exists(fwv) == 1 )
		fSamp= 1/deltax($fwv)
	endif
	Variable fsChanged= NumVarOrDefault("root:Packages:WM_IFDL:fs",1) != fSamp
	
	String dfSav= Set_IFDL_DataFolder()
	String/G freqWave= fwv
	Variable/G  fs=fSamp
	String/G    fsStr=" (sample freq = "+num2str(fs)+" Hz)"
	Variable/G  fftLen=str2num(fftPointsStr)
	Variable/G quanType=qMode
	Variable/G  quanBits=qBits
	Variable/G  ndlyCsl=csl
	String/G annoFontSizeStr=afss
	
	String/G	designList= "WMMPRLowPassDesign;";	String/G	resetFunction= "MPRLowPassReset;"
				designList += "WMMPR2BandDesign;";				resetFunction+= "MPR2BandReset;"
				designList += "WMMPR2BandNotchDesign;";			resetFunction+= "MPR2BandNotchReset;"
				designList += "WMMPR3BandDesign;";				resetFunction+= "MPR3BandReset;"
				designList += "WMMPRDiffDesign;";				resetFunction+= "MPRDiffReset;"
				designList += "WMMPRHilbertDesign;";				resetFunction+= "MPRHilbertReset;"
				designList += "WMIFDLArbGraph;";					resetFunction+= "MPRArbReset;"
				designList += "WMKaiserLowPassDesign;";			resetFunction+= "KaiserLowPassReset;"
				designList += "WMKaiserMaxFlatDesign;";			resetFunction+= "KaiserMaxFlatReset;"
				designList += "WMWindowFilterDesign;";			resetFunction+= "WindowFilterReset;"
				designList += "WMCompareFilters;"	;				resetFunction+= "Compare2FiltersReset;"

	String/G proposedFilterName
	if( strlen(proposedFilterName) == 0 )
		proposedFilterName="myFilter"
	endif
	SetDataFolder dfSav
	
	if( fSamp <= 0 )
		Abort "Sampling Frequency must be greater than zero!"
	endif

	MakeStepAndImpulse(fSamp,101)
	
	 DoWindow WMDesignPanel
	 if( V_Flag == 0 )
	 	DesignPanel()
	 endif
	if( fsChanged %| (CmpStr(GetDesignResetFunction(),"Compare2FiltersReset") == 0) )
		// Update or close any existing design graphs for new fs
		ResetCurrentDesign()
	else
		if(exists("root:Packages:WM_IFDL:designFlags") == 2) )
		 	FixCoefsDelay()	// handles changed ndlyCsl, calls StdCoefResults(1) which handles changes in quanType, quanBits
			AutoApplyFilter()
		endif
	endif
End

Proc ResetCurrentDesign()
	Silent 1; PauseUpdate
	
	String functionName= GetDesignResetFunction()
	if( exists(functionName) == 6 )
		Variable wasNormalized= 0
		if( (CmpStr(functionName,"Compare2FiltersReset") != 0) %& (exists("root:Packages:WM_IFDL:proposedFilterName") == 2) )
			String str= root:Packages:WM_IFDL:proposedFilterName
			Variable n= strlen(str)
			if (CmpStr(str[n-5,n-1],"Norm") == 0 )
				wasNormalized= 1
			endif
		endif
		Execute functionName+"()"	// recomputes the current filter; this discards the normalization
		if( wasNormalized )
			Variable nFreq=NumVarOrDefault("root:Packages:WM_IFDL:ndc_nfreq",0) * root:Packages:WM_IFDL:fs
			Variable qtzd=NumVarOrDefault("root:Packages:WM_IFDL:ndc_qtzd",1)
			NormalizeDesignCoefficients(nFreq,qtzd)
		endif
	endif
End

Function/S GetDesignResetFunction()
	SVAR designList= root:Packages:WM_IFDL:designList
	String windows= WinList("WM*",";","WIN:1")	// graphs whose names start with "WM"
	Variable n=0
	String functionName=""
	String designWindowName
	do
		designWindowName= GetStrFromList(designList, n, ";")
		if( strlen(designWindowName) == 0 )
			break
		endif
		if( FindItemInList(designWindowName, windows, ";", 0) >= 0 )
			SVAR resetFunctionsList= root:Packages:WM_IFDL:resetFunction
			functionName= GetStrFromList(resetFunctionsList,n,";")
			break	// found which design window was up (we presume that the first one found is the only one)
		endif
		n+= 1
	while (1 )
	return functionName
End

Function MakeStepAndImpulse(fs,n)
	Variable fs,n	// prefer n odd
	
	String dfSav= Set_IFDL_DataFolder()
	Make/O/N=(n) Impulse, Step
	Variable middle= trunc(n/2)
	Impulse = p == middle
	Step = p >= middle
	Variable dx= 1/fs
	Variable startX= -middle*dx
	SetScale/P x, startX, dx, "" Impulse, Step
	SetDataFolder dfSav
End

Window DesignPanel() : Panel
	PauseUpdate; Silent 1		// building window...
	
	Variable newlyInitialized=!IsIFDLInitialized()
	GuaranteeIFDLInitialized()
	if( newlyInitialized )
		// GuaranteeIFDLInitialized() called InitializeIFDL(), which calls DesignPanel() if it didn't exist
		return				// so we don't need to do it again
	endif

	DoWindow/K WMDesignPanel
	Variable x0=502
	Variable y0= 42
	Variable x1=634
	Variable y1=439
	String fmt="NewPanel/K=1/W=(%s) as \"IFDL\""
	Execute WindowCoordinatesSprintf("WMDesignPanel",fmt,x0,y0,x1,y1)
	DoWindow/C WMDesignPanel

	Variable width= x1 - x0
	Variable h= 18			// default height for a button is 20: too high
	Variable dy= h + 2		// gap between buttons
	y0 = 24				// first button y

	SetDrawLayer UserBack
	// MPR buttons
	DrawRRect 5,5,125,166
	String title="MPR Filters"
	if(  IsIFDLDemo() )
		title= "MPR Demos"
	endif
	DrawText 28,22,title
	Button mprLowPass,pos={12,y0},size={106,h},proc=ButtonMPRLowPass,title="Low Pass"
	Button mpr2Band,pos={12,y0+1*dy},size={106,h},proc=ButtonMPR2Band,title="2 Band"
	Button mpr2BandNotch,pos={12,y0+2*dy},size={106,h},proc=ButtonMPR2BandNotch,title="2 Band Notch"
	Button mpr3Band,pos={12,y0+3*dy},size={106,h},proc=ButtonMPR3Band,title="3 Band"
	Button mprDiff,pos={12,y0+4*dy},size={106,h},proc=ButtonMPRDifferentiator,title="Differentiator"
	Button mprHilbert,pos={12,y0+5*dy},size={106,h},proc=ButtonMPRHilbert,title="Hilbert"
	Button mprArb,pos={12,y0+6*dy},size={106,h},proc=ButtonMPRArbitrary,title="Arbitrary"
	// Kaiser buttons
	y0 = 169
	DrawRRect 5,y0,125,y0+3*dy
	y0 += h
	DrawText 21,y0,"Kaiser Filters"	// x,y
	title="Maximally Flat"
	if(  IsIFDLDemo() )
		title= "Max Flat Demo"
	endif
	Button kaiserMaxFlat,pos={12,y0},size={106,h},proc=ButtonKaiserMaxFlat,title=title
	Button kaiserWindowLowPass,pos={12,y0+1*dy},size={106,h},proc=ButtonKaiserWindowLowPass,title="Low Pass"
	// Window Filter buttons
	Button windowFilters,pos={12,234},size={106,h},proc=ButtonOtherWindowFilters,title="Window Filters"

	// Save, Compare buttons
	DrawLine 0,258,145,258
	Button saveCoefs,pos={16,264},size={99,20},proc=ButtonSaveDesign,title="Save Design..."
	Button compare,pos={12,287},size={110,20},proc=Compare2Filters,title="Compare Filters"

	// Apply Filter controls
	y0 = 311
	DrawLine 0,y0,145,y0
	y0 += 2
	Button applyFilter,pos={9,y0+3},size={115,20},proc=ApplyFilterButton,title="Apply Design to"
	PopupMenu dataPopup,pos={10,y0+25},size={54,19}
	PopupMenu dataPopup,mode=1,popvalue="Step",value= #"ListDataWaves()"
	PopupMenu timeFreq,pos={9,y0+45},size={121,19}
	PopupMenu timeFreq,mode=1,popvalue="Time Response",value= #"\"Time Response;Freq Response\""
	CheckBox autoApply,pos={9,y0+64},size={90,20},title="Auto Apply",value=1
	DoWindow WMCompareFilters
	SetApplyButtonTitle(V_Flag)
	SetWindow WMDesignPanel,hook=WindowCoordinatesHook
EndMacro

Function SetApplyButtonTitle(isCompare)
	Variable isCompare
	
	String newTitle
	if( isCompare )
		newTitle= "Apply Filters to"
	else
		newTitle= "Apply Design to"
	endif
	DoWindow WMDesignPanel
	if( V_Flag )
		Button applyFilter win=WMDesignPanel,title=newTitle
	endif
End

// returns path to signal wave
Function/S CheckApplyWaves(sigName,filtw,wantFreqResp)
	String sigName
	Wave filtw
	Variable wantFreqResp

	if( (CmpStr(sigName,"Step")==0) %| (CmpStr(sigName,"Impulse")==0) )
		sigName= "root:Packages:WM_IFDL:"+sigName
		NVAR fs= root:Packages:WM_IFDL:fs
		NVAR fftLen= root:Packages:WM_IFDL:fftLen
		Variable theMin= 6
		if( wantFreqResp )
			theMin= fftLen*2
		endif
		Variable n=max(theMin,2*numpnts(filtw))		// even
		MakeStepAndImpulse(fs, n)
	endif
	return sigName
End

// returns 0 if failure
Function CheckDesignApplyWaves(wantFreqResp)
	Variable wantFreqResp

	if( exists("coefs") != 1 )
		DoAlert 0, "Apply Filter: Design something, first! (coefs wave doesn't exist)"
		return 0
	endif
	Wave filtw= coefs
	ControlInfo/W=WMDesignPanel dataPopup
	String sigw= S_Value
	String sigPath= CheckApplyWaves(sigw,filtw,wantFreqResp)
	if( exists(sigPath) != 1 )
		Beep
		DoAlert 0, "Apply Filter: Signal wave "+sigw+" doesn't exist!"
		return 0
	endif
	String/G root:Packages:WM_IFDL:design_autosignal= sigPath
	return 1
End

// Returns 0 on failure
Function ApplyFilterButton(ctrlName) : ButtonControl
	String ctrlName

	Variable wantFreqResp= 0	// default to time response
	Variable wantDB= 1	// frequency response defaults to dB (overridden for differentiator)
	ControlInfo/W=WMDesignPanel timeFreq
	if( V_Flag )
		wantFreqResp= V_value-1
	endif
	if( CheckDesignApplyWaves(wantFreqResp) == 0 )
		return 0
	endif
	SVAR sigw= root:Packages:WM_IFDL:design_autosignal
	WAVE signal= $sigw
	String twoWaveNames

	DoWindow WMCompareFilters
	Variable isCompare = V_Flag	// The compare 2 filters window is up
	if( isCompare )
		WAVE filter= $CompareFilter(1)
		if( WaveExists(filter) == 0 )
			WAVE filter= $CompareFilter(2)
		else // zero or two filters exist
			WAVE filter2= $CompareFilter(2)
			if( WaveExists(filter2)  )
				twoWaveNames= ApplyFilterDontAsk(signal,filter,1,wantFreqResp,"Filt",wantDB)
				String twoMoreWaveNames= ApplyFilterDontAsk(signal,filter2,1,wantFreqResp,"Filt2",wantDB)
				String cmd=  "ShowTwoFilteredSignals("+twoWaveNames+","+twoMoreWaveNames
				cmd += ",\""+NameOfWave(filter)+"\",\""+NameOfWave(filter2)
				cmd += "\","+num2istr(wantFreqResp)+",\""+NameOfWave(signal)+"\")"
				Execute cmd
				WAVE filter=$""	// prevent ShowFilteredSignal() below from doing anything else
			endif
		endif
	else
		WAVE filter= coefs
		DoWindow WMMPRDiffDesign
		wantDB = !V_Flag			// linear frequency response for differentiator 
	endif
	twoWaveNames= ApplyFilterDontAsk(signal,filter,1,wantFreqResp,"Filt",wantDB)	// force filter on data
	if( strlen(twoWaveNames) > 0 )
		Execute "ShowFilteredSignal("+twoWaveNames+","+num2istr(wantFreqResp)+",\""+NameOfWave(signal)+"\")"
	endif
	return 1
End

// Returns 0 on failure
Function AutoApplyFilter()
	
	ControlInfo/W=WMDesignPanel autoApply
	if( V_value )	// auto apply is checked
		return ApplyFilterButton("applyFilter")
	endif
	return 0
End

Proc ButtonSaveDesign(ctrlName) : ButtonControl
	String ctrlName
	SaveDesign()
End

Function/S ListDataWaves()

	String list= WaveList("*",";","")
	list= RemoveItemFromList("coefs",list,";")
	if( exists("root:Packages:WM_IFDL:Step") == 1 )
		list= "Step;Impulse;\\M1-;" + list
	endif
	return list
End

Function/S ListFilterlikeWaves()

	String list= WaveList("*",";","")
	if( exists("coefs") == 1 )
		list= "coefs;\\M1-;"+RemoveItemFromList("coefs",list,";")	// move coefs to front, if it exists
	endif
	return list
End

Function KillAllFilterDesignWindows()

	String designList= StrVarOrDefault("root:Packages:WM_IFDL:designList","")
	
	String win
	Variable i=0
	do
		win=GetStrFromList(designList, i, ";")
		if( strlen(win) <= 0 )
			break
		endif
		DoWindow/K $win
		i += 1
	while( 1 )
	return 0
	Execute "ShowResults(0)"
End

Proc CloseAllIFDLWindows()
	Silent 1;PauseUpdate
	DoWindow/K WMDesignPanel
	DoWindow/K WMFilteredSignal
	DoWindow/K WMSineSweep
	KillAllFilterDesignWindows()
	KillReportWindows()
	ShowResults(0)
End

Proc DeleteAllIFDLData()
	Silent 1; // Delete All IFDL Data?
	DoAlert 1, "REALLY delete all IFDL Data? \r(any unsaved designs will be lost)"
	if( V_Flag == 1 )
		CloseAllIFDLWindows()
		if( DataFolderExists("root:Packages:WM_IFDL") )
			KillDataFolder root:Packages:WM_IFDL
			// are we the only package?
			String otherPackage= GetIndexedObjName("root:Packages",4,0)
			if( strlen(otherPackage) == 0 )
				KillDataFolder root:Packages
			endif
		endif
		if( DataFolderExists("root:WinGlobals") )
			DoAlert 1, "Delete the \"root:WinGlobals:\" data folder, too?"
			if( V_Flag == 1 )
				KillDataFolder root:WinGlobals
			endif
		endif
	endif
End

// assumes that at most one other left axis besides leftAxisName exists
// returns truth that traceWave was newly appended
Function AppendPassDetails(leftAxisName,traceWave)
	String leftAxisName
	Wave traceWave
	
	String graphName= WinName(0,1)
	if( (strlen(graphName) == 0) %| (WaveExists(traceWave) == 0) )
		return 0
	endif
	String detailsTraceName= NameOfWave(traceWave)
	Variable axisMin= 0
	String axes= HVAxisList(graphName,0)
	Variable axisExists= FindItemInList(leftAxisName, axes, ";", 0) >= 0 // new axis previously appended
	if( axisExists )
		axes= RemoveItemFromList(leftAxisName, axes, ";")
	endif
	String otherLeftAxis= GetStrFromList(axes, 0, ";")
	if ( strlen(AxisInfo(graphName,otherLeftAxis)) > 0 )	// there is another left axis
		ModifyGraph axisEnab($otherLeftAxis)={0,0.65}	// shrink it
		axisMin= 0.75										// and put leftAxisName above it
	endif

	// following code assumes that traceWave is not appended to any axis other than $leftAxisName
	Variable appended= 0
	CheckDisplayed traceWave
	if( V_Flag == 0 )
		AppendToGraph/L=$leftAxisName traceWave
		appended= 1
		ModifyGraph lblPos($leftAxisName)=60
		ModifyGraph lblLatPos($leftAxisName)=-5
		ModifyGraph freePos($leftAxisName)={0,bottom}
		ModifyGraph axisEnab($leftAxisName)={axisMin,1}
		ModifyGraph nticks($leftAxisName)=2
		ModifyGraph rgb($detailsTraceName)=(0,0,0)
		if( strlen(WaveUnits(traceWave,-1)) > 0 )
			Label $leftAxisName "details (\U)"
		else
			Label $leftAxisName "details"
		endif
		SetAxis/A/N=1/E=0 $leftAxisName
	endif
	return appended
End

Proc PMPR2BandNoShow(islp,nt,b1end,b2start,b1weight)
	Variable islp,nt,b1end,b2start,b1weight
	
	String dfSav= Set_IFDL_DataFolder()
	String/G designTypeName="McClellan-Parks; two band"
	Variable/G designFlags= 0x3
	Make/O bandInfo= {islp,0,b1end,!islp,b2start,0.5*fs}
	b1end /= fs
	b2start /= fs
	if( ChkFreq(b1end,b2start,0,0) )
		SetDataFolder dfSav
		abort
	endif
	String mpDes="des"
	String mpWt="wt"
	String mpGrid="grid"
	MPRMakeWaves(mpDes,mpWt,mpGrid,nt)
	$mpDes(0,b1end)=islp
	$mpDes(b2start,)=islp==0
	$mpWt(0,b1end)=b1weight
	$mpGrid(b1end,b2start)=NaN

	SetDataFolder dfSav
	Make/O/N=(nt) coefs
	MPKernel(mpDes,mpWt,mpGrid,"coefs",0)
	StdCoefsTreatmentNoShowResults(0x1e,1)
End

Proc PMPR2Band(islp,nt,b1end,b2start,b1weight)
	Variable islp,nt,b1end,b2start,b1weight
	PMPR2BandNoShow(islp,nt,b1end,b2start,b1weight)
	ShowResults(0x1e)
End

Function MaximumInBand(w,bandStartX,bandEndX)
	Wave w
	Variable bandStartX,bandEndX
	WaveStats/Q/R=(bandStartX,bandEndX) w
	return V_max
End

Function dBIsChecked(graphName)
	String graphName
	
	DoWindow $graphName
	if( V_Flag == 1 ) 							// window exists, so get state of dB Response checkbox
		ControlInfo/W=$graphName dbCheck	// requires that the name of the dbCheck control be dbCheck in $graphName window
		return V_value 
	endif
	return 0	// window doesn't exist
End

// this shouldn't be called directly, because the other traces
// on the graph need to be adjusted, as will the legend.
// Rather, call dBCheck() as part of other operations that adjust the graph
Function dBCheck(isdB)
	Variable isdB

	Wave wv= root:Packages:WM_IFDL:coefsMag
	Wave dbwv= root:Packages:WM_IFDL:coefsdBMag
	SwapResponseLeftWaves(wv,dbwv,isdB)
End

Function SwapResponseLeftWaves(w1,w2,want_w2)
	Wave w1,w2
	Variable want_w2
	
	if( want_w2 )
		CheckDisplayed w1
		if( V_Flag %& WaveExists(w2) )
			String wvName= NameOfWave(w1)
			ReplaceWave trace=$wvName w2
			Label/Z responseLeft "response (dB)"
		endif
	else
		CheckDisplayed w2
		if( V_Flag %& WaveExists(w1) )
			String dbwvName= NameOfWave(w2)
			ReplaceWave trace=$dbwvName w1
			Label/Z responseLeft "response"
		endif
	endif
End


Function designHook(infoStr)
	String infoStr

	WindowCoordinatesHook(infoStr)
	Variable statusCode= 0
	String event= StrByKey("EVENT",infoStr)
	if( CmpStr(event,"kill") == 0 )
		statusCode= 1
		Execute "ShowResults(0)"
	endif
	return statusCode
end


Function WMMoveTraceToBottom(graphName,traceName)
	String graphName,traceName

	DoWindow $graphName
	if( V_Flag == 0 )
		return 0		// named graph doesn't exist
	endif
	String sep= ","
	String currentTraces= TraceNameList(graphName,sep,1)

	if( FindItemInList(traceName,currentTraces,sep,0) < 0 )
		return 0		// trace not in graph
	endif	
	
	String bottomTrace= GetStrFromList(currentTraces, 0, sep)
	if( CmpStr(bottomTrace,traceName) == 0 )
		return 1		// trace already at bottom
	endif
	
	String cmd
	Sprintf cmd, "ReorderTraces %s, {%s}"bottomTrace, traceName
	Execute cmd // move to back
End

// mpDes,mpWt,mpGrid are in the WM_IFDL data folder, but don't have the IFDL path in them
// cwave is in the current data folder
Proc MPKernel(mpDes,mpWt,mpGrid,cwave,neg)
	String mpDes,mpWt,mpGrid,cwave		// desired response, weighting, frequency grid, coefs results
	Variable neg							// 1 => anti-symetric type filter

	if( IsIFDLDemo() )
		LoadDemoFilter(DemoFilterName(),cwave)
		return
	endif
	
	Variable nt= numpnts($cwave)

	String dfSav= Set_IFDL_DataFolder()
	Sort $mpGrid,$mpDes,$mpWt,$mpGrid
	WaveStats/Q $mpGrid
	DeletePoints V_npnts,1e6,$mpDes,$mpWt,$mpGrid
	
	String df= GetDataFolder(1)
	SetDataFolder dfSav	// so that an error during Remez doesn't leave the data folder in the wrong place
	
	Remez/N=(neg)/Q $(df+mpDes),$(df+mpWt),$(df+mpGrid),$cwave	// use /Q=3 to stop on all serious errors
	if( (V_Flag > 0) %& (V_Flag < 4) )
		DoAlert 0, "The filter design did not converge after "+num2istr(V_Flag)+" iterations. Please change the design parameters and try again."
	endif
	KillWaves $(df+mpDes),$(df+mpWt),$(df+mpGrid)
	Variable pTmp=floor(nt/2)
	neg=1-2*neg
	if(mod(nt,2)!=0)
		$cwave[pTmp+1,]= neg*$cwave[2*pTmp-p]	// MPR algorithm gives 1/2 the coefficients...
	else
		$cwave[pTmp, ]= neg*$cwave[2*pTmp-p-1]	// ... make the rest by symmetry
	endif

End

Proc IFDLKaiserMaxFlat(beta,gamma,cfs)
	Variable beta,gamma
	String cfs
	if( IsIFDLDemo() )
		LoadDemoFilter(DemoFilterName(),cfs)	// replaces cfs with the pre-built Kaiser MaxFlat coefficient wave
	else
		FMaxFlat beta,gamma,$cfs
		Variable ncoefs=$(cfs)[0]	// half the actual number of coefficients
		Redimension/N=(2*ncoefs-1) $cfs	// create the other half by symmetry
		Rotate ncoefs-2,$cfs
		$(cfs)[0,ncoefs-2]=$(cfs)[2*ncoefs-2-p]
	endif
End

Function/S DemoFilterName()
	String filterName=""
	
	if( exists("root:Packages:WM_IFDL:proposedFilterName") == 2 )
		SVAR str= root:Packages:WM_IFDL:proposedFilterName
		Variable n= strlen(str)
		if (CmpStr(str[n-5,n-1],"Norm") == 0 )
			filterName= str[0,n-6]	// ignore trailing "Norm"
		else
			filterName= str
		endif
	endif
	return filterName
End

// outputName is name of wave in current data folder
Function LoadDemoFilter(filterName,outputName)
	String filterName,outputName
	
	String dfSav= Set_IFDL_DataFolder()
	if( exists(filterName) != 1 )
		LoadWave/O/P=Igor ":IFDL Procedures:Demo Filters:"+filterName+".bwav"
	endif
	if( exists(filterName) == 1 )
		Print "Loading demo filter"+filterName+" into "+outputName
		Duplicate/O $filterName $(dfSav+outputName)
		DoAlert 0, "Demo filter \""+filterName+"\" uses only the design defaults. Contact WaveMetrics to purchase the full IFDL package."
	else
		DoAlert 0,  "Demo filter \""+filterName+"\" not found; using previous filter design."
	endif
	SetDataFolder dfSav
End

// creates the named waves in the IFDL data folder
Proc MPRMakeWaves(mpDes,mpWt,mpGrid,clen)
	String mpDes,mpWt,mpGrid
	Variable clen							// length of corresponding coef wave

	Variable glen=(clen+1)*8				// put your own (grid density/2) here
	if( glen > 80000 )						// Remez will not handle any more than this.
		DoAlert 1, num2istr(clen)+" coefficents exceeds limit of 9999.\rLimit to 9999?"
		if( V_Flag != 1 )
			return
		endif 
		glen= 80000
	endif
	String dfSav= Set_IFDL_DataFolder()
	Make/O/N=(gLen) $mpDes,$mpWt,$mpGrid
	SetScale x,0,0.5*(1+gLen)/(gLen),$mpDes,$mpWt,$mpGrid
	$mpGrid=x
	$mpDes=0
	$mpWt=1
	SetDataFolder dfSav
End


Proc NormalizeDesignCoefficients(nFreq,qtzd)	// Normalize coefficients according to frequency response
	Variable nFreq=NumVarOrDefault("root:Packages:WM_IFDL:ndc_nfreq",0) * NumVarOrDefault("root:Packages:WM_IFDL:fs",1)
	Variable qtzd=NumVarOrDefault("root:Packages:WM_IFDL:ndc_qtzd",1)
	Prompt nFreq "normalization frequency, or <0 for peak frequency"
	Prompt qtzd "quantize the coefficents?", popup, "No;Yes"

	PauseUpdate; Silent 1		// Normalize Design Coefficients...
	GuaranteeIFDLInitialized()
	nFreq /= root:Packages:WM_IFDL:fs	
	Variable/G root:Packages:WM_IFDL:ndc_nfreq= nFreq
	Variable/G root:Packages:WM_IFDL:ndc_qtzd= qtzd
	if( nFreq > 0.5)
		abort "frequency may not exceed "+num2str(root:Packages:WM_IFDL:fs/2)+" Hz"
	endif
	if( exists("root:Packages:WM_IFDL:preQuantCoefs") != 1 )
		abort "Design something, first!"
	endif
	Variable response=CalcNormalizingResponse(root:Packages:WM_IFDL:preQuantCoefs,nFreq) // response at nFreq or at peak
	if( response <= 0 )
		abort "Response was 0 at that frequency! Normalization aborted."
	endif
	root:Packages:WM_IFDL:preQuantCoefs /= response
	Variable/G root:Packages:WM_IFDL:ndc_norm= response			// for undo, someday?
	if( exists("root:Packages:WM_IFDL:proposedFilterName") == 2 )
		String str= root:Packages:WM_IFDL:proposedFilterName
		Variable n= strlen(str)
		if (CmpStr(str[n-5,n-1],"Norm") != 0 )
			root:Packages:WM_IFDL:proposedFilterName+="Norm"
		endif
	endif
	String info
	sprintf info "coefs /= %.15g", response
	Note coefs, info	// appends to existing note
	Print info
	StdCoefResults(qtzd == 2)	// transfers preQuantCoefs to coefs
	DoWindow WMDesignPanel
	if( V_Flag)
		AutoApplyFilter()
	endif
EndMacro

Function RemoveAllTraces()
	Execute "RemoveFromGraph "+RemoveLastChar( TraceNameList("",",",1))
End

Function FilteredWindowChanged(expectedNote,wSigName)
	String expectedNote,wSigName

	DoWindow/F WMFilteredSignal
	if( V_Flag == 1 )	// window exists
		GetWindow WMFilteredSignal note	// in S_Value
		if( CmpStr(S_Value,expectedNote) == 0 )
			return 0	// no changes to existing window
		endif
		RemoveAllTraces()
	else
		// build the window
		Variable x0=6*72 / ScreenResolution,   y0=355*72 / ScreenResolution
		Variable x1=490*72 / ScreenResolution, y1=474*72 / ScreenResolution
		String fmt="Display/K=1/W=(%s)"
		Execute WindowCoordinatesSprintf("WMFilteredSignal",fmt,x0,y0,x1,y1)
		DoWindow/C/T WMFilteredSignal "Filtered "+wSigName
	endif
	
	DoWindow/T WMFilteredSignal "Filtered "+wSigName
	DoUpdate	// needed for some Windows versions of Igor
	return 1
End

// Changes to the Filtered Data graph needed by both ShowFilteredSignal and ShowTwoFilteredSignals
Function/S ShowFilterStyleLegend(wSigName, dataName,wantFreqResp)
	String wSigName, dataName
	Variable wantFreqResp
	
	PauseUpdate
	ModifyGraph lowTrip(bottom)=0.0001	// prevent milliseconds from being displayed in scientific notation.
	ModifyGraph minor(bottom)=1
	ModifyGraph rgb($wSigName)=(0,0,0), lstyle($wSigName)=2
	if( (wantFreqResp==0) %& (CmpStr(wSigName,"Impulse") == 0) )
		ModifyGraph mode($wSigName)=1
	endif
	Label left, ""
	String sigLegend=DesignLegendSize()+"\\s("+wSigName+") "+dataName
	if( wantFreqResp )
		sigLegend += " Freq Response"
		if( StringEndsWith("dBMag",wSigName) )
			Label left, "dB"
		endif
	else
		if( CmpStr(wSigName,"Impulse") == 0 )
			ModifyGraph mode($wSigName)=1
		endif
	endif
	return sigLegend
End

// wSig is original data (time or frequency)
// wFiltered is filtered data (time or frequency)
// The window is rebuilt only if wSig,wFiltered, and wantFreqResp are different than last time (stored in the window note)
Function ShowFilteredSignal(wSig,wFiltered,wantFreqResp,dataName)
	Wave wSig,wFiltered
	Variable wantFreqResp
	String dataName
	
	String expectedNote
	Sprintf expectedNote, "wSig=%s, wFiltered=%s, wantFreqResponse=%d, data=%s",GetWavesDataFolder(wSig,2),GetWavesDataFolder(wSig,2),wantFreqResp,dataName
	expectedNote+="filteredDelay="+num2str(leftx(wFiltered))
	String wSigName= NameOfWave(wSig)

	if( FilteredWindowChanged(expectedNote,dataName) == 0 )	// no changes
		return 0	// no changes to existing window
	endif

	AppendToGraph wSig,wFiltered	// filtered above signal
	String wFilteredName= NameOfWave(wFiltered)
	ModifyGraph rgb($wFilteredName)=(0,0,65535)
	Legend/C/N=filterLegend ShowFilterStyleLegend(wSigName, dataName,wantFreqResp)
	AppendText DesignLegendSize()+"\\s("+wFilteredName+") Filtered "+dataName
	SetWindow WMFilteredSignal,hook=WindowCoordinatesHook,note=expectedNote
	return 1 // changed the window
End


// wSig is original data (time or frequency)
// wFiltered is filtered data (time or frequency)
// The window is rebuilt only if the inputs are different than last time (stored in the window note)
Function ShowTwoFilteredSignals(wSig,wFiltered,wSigAgain,wFiltered2,filter1Name,filter2Name,wantFreqResp,dataName)
	Wave wSig,wFiltered,wSigAgain,wFiltered2
	String filter1Name,filter2Name,dataName
	Variable wantFreqResp

	String expectedNote
	Sprintf expectedNote, "wSig=%s, wFiltered=%s, wFiltered2=%s, wantFreqResponse=%d, filters="+filter1Name+filter2Name,GetWavesDataFolder(wSig,2),GetWavesDataFolder(wFiltered,2),GetWavesDataFolder(wFiltered2,2),wantFreqResp
	expectedNote+="filteredDelay1="+num2str(leftx(wFiltered))+"filteredDelay2="+num2str(leftx(wFiltered2))
	String wSigName= NameOfWave(wSig)

	if( FilteredWindowChanged(expectedNote,dataName) == 0 )	// no changes
		return 0	// no changes to existing window
	endif

	AppendToGraph wSig,wFiltered,wFiltered2	// filtered above signal
	String wFilteredName= NameOfWave(wFiltered)
	String wFiltered2Name= NameOfWave(wFiltered2)
	ModifyGraph rgb($wFiltered2Name)=(0,0,65535)	// same color as filter response in compare design window
	Legend/C/N=filterLegend ShowFilterStyleLegend(wSigName, dataName,wantFreqResp)
	AppendText DesignLegendSize()+"\\s("+wFilteredName+") "+filter1Name+" Response"
	AppendText DesignLegendSize()+"\\s("+wFiltered2Name+") "+filter2Name+" Response"
	SetWindow WMFilteredSignal,hook=WindowCoordinatesHook,note=expectedNote
	return 1 // changed the window
End

Function/S FilterConvolve(sw,dw)	// uses the built-in Convolve operation for acausal convolution
	Wave sw,dw
	
	if( leftx(sw) < 0 )
		Convolve/A sw,dw
	else
		Variable n= numpnts(dw)
		Variable halfFilterLen= trunc((numpnts(sw)-1)/2)
		Convolve sw,dw
		Redimension/N=(n+halfFilterLen) dw
	endif
	SetScale/P x, leftx(dw), deltax(sw), "" dw	// impose filter timing on data
	return GetWavesDataFolder(dw,2)
end


// Returns two full paths of time-domain or frequency-domain output waves that should be displayed
// if liberal names, they are quoted.
// or "" on error
Function/S ApplyFilterDontAsk(sigw,coefw,imposeFilterOnData,wantFreqResp,suffix,wantDB)
	Wave sigw,coefw
	Variable imposeFilterOnData
	Variable wantFreqResp,wantDB
	String suffix

	if( (WaveExists(sigw) == 0) %| (WaveExists(coefw) == 0) )	
		return ""
	endif

	if(imposeFilterOnData != 0)
		SetScale/P x leftx(sigw),deltax(coefw),sigw	// use filter X scaling
	endif
	
	String sigPath=    GetWavesDataFolder(sigw,2)
	String resultPath= DupWaveInDataFolder(sigw,suffix)
	String/G root:Packages:WM_IFDL:af_timeResult= resultPath

	WAVE result= $resultPath

	FilterConvolve(coefw,result)
	if( wantFreqResp )
		sigPath=    FreqMagnitude(sigw,wantDB,0)
		resultPath= FreqMagnitude(result,wantDB,0)
		String/G root:Packages:WM_IFDL:af_freqSig= sigPath
		String/G root:Packages:WM_IFDL:af_freqResult= resultPath
	endif
	return sigPath+","+resultPath
End

// Returns "" on failure
Function/S ApplyFilterAsk(sigw,coefw,wantFreqResp,wantDB)
	Wave sigw,coefw
	Variable wantFreqResp,wantDB
	
	// Verify frequency match
	Variable freq= 1/deltax(coefw)	// filter frequency
	Variable dx=deltax(sigw)			// data X scaling
	Variable imposeFilterOnData = 0 				// don't change data's sampling frequency unless mismatch exceeds 0.1%, and user allows it.
	if((freq*dx<.999)+(freq*dx>1.001)) // mismatch exceeds +/- .1%
		String dataName= NameOfWave(sigw)
		String str
		Sprintf str, "WARNING: %s sampling frequency = %g Hz\r(filter frequency = %g Hz)\rForce %s to %g Hz?",dataName,1/dx,freq,dataName,freq
		DoAlert 3, str
		if( V_Flag == 3 )
			return ""
		endif
		imposeFilterOnData= V_Flag == 1
	endif
	
	String resultNames= ApplyFilterDontAsk(sigw,coefw,imposeFilterOnData,wantFreqResp,"Filt",wantDB)
	return resultNames
End

Proc ApplyFilter(sigw,force1Warn2,coefw,time1Freq2FreqDB3)	
	String sigw=StrVarOrDefault("root:Packages:WM_IFDL:af_sig","")
	Variable force1Warn2=NumVarOrDefault("root:Packages:WM_IFDL:af_forceWarn",1)
	String coefw=StrVarOrDefault("root:Packages:WM_IFDL:af_coefs","coefs")
	Variable time1Freq2FreqDB3=NumVarOrDefault("root:Packages:WM_IFDL:af_timeFreqDB",1)
	Prompt sigw "data wave",popup ListDataWaves()
	Prompt force1Warn2 "data wave sample frequency"+StrVarOrDefault("root:Packages:WM_IFDL:fsStr",""),popup,"force to samp freq;warn if not correct"
	Prompt coefw "filter coefficients",popup ListFilterLikeWaves()
	Prompt time1Freq2FreqDB3 "show filtered result as",popup, "Time Response;Frequency Response (Linear);Frequency Response (dB)"

	PauseUpdate;Silent 1	// Apply Filter...
	GuaranteeIFDLInitialized()
	String dfSav= Set_IFDL_DataFolder()
	String/G   af_sig=sigw
	Variable/G af_forceWarn=force1Warn2
	String/G   af_coefs=coefw
	Variable/G af_timeFreqDB=time1Freq2FreqDB3
	Variable freq= fs
	SetDataFolder dfSav
	
	Variable imposeFilterOnData= force1Warn2 == 1	// else, just warn the user
	Variable wantFreqResp= time1Freq2FreqDB3 > 1
	Variable wantDB= time1Freq2FreqDB3 == 3

	// sigw could be "Step" or "Impulse" from the WM_IFDL data folder.
	String sigPath=  CheckApplyWaves(sigw,$coefw,wantFreqResp)
	if( exists(sigPath) != 1 )
		Abort sigw+ "doesn't exist!
	endif

	Variable dx=deltax($sigPath)
	if((freq*dx<.999)+(freq*dx>1.001))	// +/- .1%
		if( imposeFilterOnData )
			SetScale/P x leftx($sigPath),1/freq,$sigPath
			Print sigw+" samp freq forced"+root:Packages:WM_IFDL:fsStr
		else
			String str
			Sprintf str, "WARNING:  %s samp freq = %g Hz\r%s",sigw,1/dx,root:Packages:WM_IFDL:fsStr
			DoAlert 0,str
		endif
	endif
	
	String twoWaveNames= ApplyFilterDontAsk($sigPath,$coefw,imposeFilterOnData,wantFreqResp,"Filt",wantDB)
	if( strlen(twoWaveNames) > 0 )
		Execute "ShowFilteredSignal("+twoWaveNames+","+num2istr(wantFreqResp)+",\""+sigw+"\")"
		if( CmpStr(coefw,"coefs") != 0 )
			AppendText "Filter: "+coefw
		endif
	endif
End	


Proc CreateReport()
	
	PauseUpdate;Silent 1	// Create Report
	if( exists("coefs") != 1 )
		Abort "Design something, first!"
	endif
	GuaranteeIFDLInitialized()

	String reportBody,stmp,coefsStr
	Variable pErr,nErr,ncoefs=numpnts(coefs),nrows

	DoWindow/F WMCoefsTable
	if(V_Flag==0)
		CoefsTable()
	endif
	if(ncoefs>63)		// max rows possible with smallest font size
		if(ncoefs<=126)
			nrows=ceil(ncoefs/2);coefsStr="; first half shown"	// just show the first half
		else
			nrows=63;coefsStr="; first 63 shown"
		endif
	else
		nrows=ncoefs
	endif
	if(nrows>53)
		ModifyTable size=7		// more than 53 rows, must use 7 point
	else
		if(nrows>45)
			ModifyTable size=8
		else
			ModifyTable size=9	// max 45 rows with 1" margins on US 11" page
		endif
	endif
	
	Variable fs= root:Packages:WM_IFDL:fs
	Variable qb= root:Packages:WM_IFDL:quanBits
	Variable dflgs= root:Packages:WM_IFDL:designFlags
	
	sprintf reportBody,"\\Z08Sampling frequency:  %gHz\rNum terms: %d%s\rQuantization: ",fs,ncoefs,coefsStr
	Variable digits=limit(ceil(qb/3.322),3,16)
	if(root:Packages:WM_IFDL:quanType == 1)
		reportBody+="Floating point"
		ModifyTable format(coefs)=5,digits(coefs)=(digits)
	else
		reportBody+="Fixed point"
		ModifyTable format(coefs)=3,digits(coefs)=(digits)
	endif
	sprintf stmp,", %d bits",qb
	reportBody+=stmp
	
	if(dflgs %& 0x2)				//  calculate ripple?
		Variable i= 0;
		do
			sprintf stmp,"\rBand %d: ",i/3 + 1
			reportBody+=stmp
			WaveStats/Q/R=(root:Packages:WM_IFDL:bandInfo[i+1],root:Packages:WM_IFDL:bandInfo[i+2]) root:Packages:WM_IFDL:coefsMag
			if(root:Packages:WM_IFDL:bandInfo[i]==0)		// stop band
				sprintf stmp,"stopband[%g,%g Hz], min reject %.1fdB",root:Packages:WM_IFDL:bandInfo[i+1],root:Packages:WM_IFDL:bandInfo[i+2],20*log(V_max)
			else
				pErr=20*log(V_max)
				nErr=20*log(V_min)
				sprintf stmp,"passband[%g,%g Hz], peak error +%.3f,%.3fdB",root:Packages:WM_IFDL:bandInfo[i+1],root:Packages:WM_IFDL:bandInfo[i+2],pErr,nErr
			endif
			reportBody+=stmp
			i+= 3
		while( i < numpnts(root:Packages:WM_IFDL:bandInfo) )
	endif

	ShowResults(root:Packages:WM_IFDL:dispMode)

	if(dflgs%&1)
		DoWindow/K WMIFDLLayout0
		DoWindow/K WMIFDLLayout2
		DoWindow/F WMIFDLLayout1
		if(V_Flag==0)
			IFDL_Layout1()
		endif
	else
		if(dflgs%&4)		// must be differentiator
			DoWindow/K WMIFDLLayout0
			DoWindow/K WMIFDLLayout1
			DoWindow/F WMIFDLLayout2
			if(V_Flag==0)
				IFDL_Layout2()
			endif
			sprintf stmp,"\rDesired slope: %g",root:Packages:WM_IFDL:bandInfo[0]
			reportBody+=stmp
		else
			DoWindow/K WMIFDLLayout1
			DoWindow/K WMIFDLLayout2
			DoWindow/F WMIFDLLayout0
			if(V_Flag==0)
				IFDL_Layout0()
			endif
		endif
	endif
	ModifyLayout rows(WMCoefsTable)=nrows
	ReplaceText/N=title "\JC"+root:Packages:WM_IFDL:designTypeName
	ReplaceText/N=specs reportBody
End

Proc  KillReportWindows()
	Silent 1;PauseUpdate
	DoWindow/K WMIFDLLayout0
	DoWindow/K WMIFDLLayout1
	DoWindow/K WMIFDLLayout2
	DoWindow/K WMCoefsTable
End
	
Proc SaveDesign(format)
	Variable format=4
	Prompt format, "Save Design as:",popup,"Descriptive Text File;C Include File;Igor Binary File;Another Igor Wave"
	
	if( format == 1 )
		DescriptiveTextFile()
	endif
	if( format == 2 )
		CIncludeFile()
	endif
	if( format == 3 )
		IgorBinaryFile()
	endif
	if( format == 4 )
		AsAnotherWave()
	endif
End

Function/S ProposedName()
	return StrVarOrDefault("root:Packages:WM_IFDL:proposedFilterName","myFilter")
End

Proc DescriptiveTextFile()

	Silent 1		// Save Descriptive Text File ...
	if( exists("coefs") != 1 )
		Abort "Couldn't find \"coefs\" wave in current data folder!"
	endif
	GuaranteeIFDLInitialized()
		
	Variable oFile,ncoefs=numpnts(coefs)
	
	Open/M="General Format Coefs" ofile
	duplicate/o coefs intCoefs,coefIndex,absIntCoefs
	intCoefs=round(coefs*2^(root:Packages:WM_IFDL:quanBits-1))
	Redimension/D absIntCoefs	// need 32 bits of mantissa for %8X and negative coefficients.
	absIntCoefs= (intCoefs[p] < 0 ) * (2^32 + intCoefs[p]) + (intCoefs[p] >= 0 ) * intCoefs[p]
	coefIndex= p
	Make/D/O/N=(ncoefs) coefsFP
	coefsFP= FormFloatBits(coefs,8,23)
	
	fprintf oFile,"   FILTER COEFFICIENT FILE \r FIR DESIGN\r"
	fprintf oFile,"%4d%76s\r",ncoefs,	"/* number of taps in decimal */"
	fprintf oFile,"%.4x%76s\r",ncoefs,	"/* number of taps in hexadecimal*/"
	fprintf oFile,"%2d%78s\r",root:Packages:WM_IFDL:quanBits,	"/* number of bits in quantized coefficients(decimal) */"
	fprintf oFile,"%.4x%76s\r",root:Packages:WM_IFDL:quanBits,	"/* number of bits in quantized coefficients(hexadecimal)*/"
	fprintf oFIle," 0  %50s\r","/* shift count in decimal */"
	fprintf oFIle,"0000%50s\r","/* shift count in hexadecimal */"
	wfprintf oFile, "%11d %.8X  /* coefficient of tap %-8d*/\r",intCoefs,absIntCoefs,coefIndex
	wfprintf oFile, "%24.15e %.8X00000000   /* coefficient of tap %-8d*/\r",coefs,coefsFP,coefIndex
	Close oFile
	KillWaves intCoefs,coefIndex,coefsFP,absIntCoefs
End

Proc CIncludeFile()
	
	Silent 1		// Save C Include File...
	if( exists("coefs") != 1 )
		Abort "Couldn't find \"coefs\" wave in current data folder!"
	endif
	GuaranteeIFDLInitialized()
	Variable oFile,ncoefs=numpnts(coefs),ncols=5,k
	Open/M="C Format Coefficients" ofile
	
	fprintf oFile,"/* FIR filter coefficients */\r\rstatic double coefs[]={"
	k=0
	do
		if(k==(ncoefs-1))	// last data point?
			fprintf oFile,"%g\r};\r\r",coefs[k]
			break
		endif
		if(mod(k,ncols)==0)
			fprintf oFile,"\r\t"
		endif
		fprintf oFile,"%g,",coefs[k]
		k+=1
	while(1)

	Close oFile
End
	
Proc IgorBinaryFile(wn)
	String wn=ProposedName()
	Prompt wn,"save \"coefs\" wave with wave name:"
	
	Silent 1		// Save Igor Binary File...
	if( exists("coefs") != 1 )
		Abort "Couldn't find \"coefs\" wave in current data folder!"
	endif
	GuaranteeIFDLInitialized()
	wn= CleanupName(wn,1)
	// save in the IFDL data folder to prevent saving over an
	// existing saved filter saved with AsAnotherWave()
	String dfSav= Set_IFDL_DataFolder()
	String cwave= dfSav+"coefs"
	if (CmpStr(wn,"coefs") != 0)
		Duplicate/O $cwave,$wn
		Save/I $wn 
		KillWaves $wn
	else
		Save/I $cwave 
	endif
	SetDataFolder dfSav
End

Proc AsAnotherWave(wn,overWrite)
	String wn=ProposedName()
	Variable overWrite=NumVarOrDefault("root:Packages:WM_IFDL:scc_overwrite",1)
	Prompt wn,"copy \"coefs\" to new wave named:"
	Prompt overWrite," Overwrite any existing wave:",popup,"Yes;No"

	Silent 1		// Save a copy of "coefs" wave under another name...
	if( exists("coefs") != 1 )
		Abort "Couldn't find \"coefs\" wave in current data folder!"
	endif
	GuaranteeIFDLInitialized()
	wn= CleanupName(wn,1)
	String/G root:Packages:WM_IFDL:scc_wn= wn
	Variable/G root:Packages:WM_IFDL:scc_overwrite= overWrite
	if( (exists(wn) == 1) %& (overWrite == 2) )
		DoAlert 0, wn+" already exists!\r(Not overwritten, \"coefs\" not saved.)"
	else
		if (CmpStr(wn,"coefs") != 0)
			Duplicate/O coefs,$wn
		else
			DoAlert 0, "Saving a wave as \"coefs\" won't help you: \"coefs\" will be overwritten by the next design!"
		endif
	endif
End


Function REdge(xx,x0,delta,isRise)		// rounded edge
	Variable xx,x0,delta,isRise			// x0 is the center of the edge and delta is the 10-90% rise time.
	
	if(delta==0)
		if(isRise)
			return xx>x0
		else
			return xx<x0
		endif
	endif
	Variable t=0.5*(1+tanh(2.19722*(xx-x0)/delta))
	if(isRise)
		return t
	else
		return 1-t
	endif
end


Function/S DupWaveInDataFolder(w,suffix)
	Wave w
	String suffix
	String dfSav = GetDataFolder(1)
	String wavesDF= GetWavesDataFolder(w,1) 
	SetDataFolder wavesDF
	String dupName= CleanupName(NameOfWave(w)+suffix,1)		// liberal names allowed
	Duplicate/O w, $dupName
	Wave dup= $dupName
	SetDataFolder dfSav
	return GetWavesDataFolder(dup,2)
End

Function/S FreqMagnitude(sw,dB,doWin)
	Wave sw
	Variable dB,doWin
	
	String sfx
	if( dB )
		sfx= "DbMag"
	else
		sfx="Mag"
	endif	
	String swPath= GetWavesDataFolder(sw,2)
	String owPath= DupWaveInDataFolder(sw,sfx)

	WAVE ow= $owPath
	NVAR fftLen= root:Packages:WM_IFDL:fftLen

	Variable np=2*trunc((numpnts(sw)+1)/2)	// need even number of points
	np=max(2*fftLen,np)								// output is a minimum of 128 points long

	if( doWin )
		Hanning	 ow	// appropriate for input data, but not freq response of filter
	endif
	
	Redimension/N=(np) ow		// appends zeros
	FFT ow
	WAVE/C owc= ow
	owc= r2polar(owc[p])
	Redimension/R owc
	WAVE ow= owc

	if( dB )
		ow= 20*log(ow[p])
	endif
	return owPath
end

Function/D FormFloatBits(anum,eBits,mBits)
	Variable anum,eBits,mBits		// bits in the exponent, and in the mantissa ( not including hidden 1)
	
	Variable isNeg=anum<0,hasHidden=1		// use 0 if number format does not use the hidden bit trick
	Variable/D exponent=0,mantissa=0,mask
	anum=abs(anum)
	if(anum==0)
		return 0
	endif
	exponent=ceil(log2(anum))
	mantissa=anum*2^(mBits+hasHidden-exponent-1)
	if((mantissa%&2^(mBits+hasHidden-1))==0)
		mantissa *= 2
		exponent-=1
	endif
	mantissa=round(mantissa)
	exponent+=2^(eBits-1)-1				// assume offset binary -- you may code other formats
	mask=2^(mBits+hasHidden-1)-1		// make a mask -- only necesary if hidden bit
	return  isNeg*2^(mBits+eBits)%|(exponent*2^mBits)%|(mantissa%&mask)
end

Function ApplyQuant(src,dest,qType,qBits)
	wave src,dest
	Variable qType,qBits
	
	Variable np=numpnts(src),ii,scale,big,mf
	CopyScales/P src,dest
	
	if((qType==1)%&(qBits==24))	// default mode
		ii=0
		do
			dest[ii]=src[ii]
			ii+=1			// straight transfer
		while(ii<np)
		return 0
	endif
	if(qType==1)
		ii=0
		do
			if(src[ii]==0)
				dest[ii]=src[ii]
			else
				mf=2^(qBits-floor(log2(abs(src[ii]))))
				dest[ii]=round(src[ii]*mf)/mf
			endif
			ii+=1
		while(ii<np)
	else
		ii=1
		big=abs(src[0])
		do
			if(abs(src[ii])>big)
				big=abs(src[ii])
			endif
			ii+=1
		while(ii<np)
		scale=2^(qBits-1)-1			// max pos integer in a 2's comp qBit word
		ii=0
		do
			dest[ii]=(round((src[ii]/big)*scale)/scale)*big
			ii+=1
		while(ii<np)
	endif
	return 0
end

// returns a magnitude response value
Function CalcNormalizingResponse(w,freq)
	Wave w			// coefficient wave
	Variable freq	// normalized frequency for magnitude or -1 for peak magnitude
	
	Variable n=4*CeilPwr2(numpnts(w))	// interpolate by factor of 4
	String dfSav= Set_IFDL_DataFolder()
	Duplicate/O w wcn_srctmp
	Redimension/N=(n) wcn_srctmp
	fft wcn_srctmp
	Wave/C wcmplx= wcn_srctmp
	n=numpnts(wcmplx)
	Make/O/N=(n) wcn_dw
	Variable sum
	if(freq < 0)
		wcn_dw=magsqr(wcmplx)
		WaveStats/Q wcn_dw
		sum=sqrt(V_Max)
	else
		sum=sqrt(magsqr(wcmplx[freq*2*(n-1)]))
	endif
	KillWaves wcmplx,wcn_dw
	SetDataFolder dfSav
	return sum
end

Function/S Set_IFDL_DataFolder()
	String dfSav= GetDataFolder(1)
	NewDataFolder/O/S root:Packages
	NewDataFolder/O/S WM_IFDL
	return dfSav
End

// Returns true if error!
Function ChkFreq(f1,f2,f3,f4)
	Variable f1,f2,f3,f4
	Variable noGood= 0
	if((f1>0.5)+(f2>0.5)+(f3>0.5)+(f4>0.5)+(f1<0)+(f2<0)+(f3<0)+(f4<0))
		NVAR fs = root:Packages:WM_IFDL:fs
		DoAlert 0,"Frequencies must be in range 0.."+num2str(fs/2)+"Hz"
		noGood= 1
	endif
	return noGood
End

// expects coefs wave in current data folder.
Proc StdCoefsTreatmentNoShowResults(mode,wantResults)
	Variable mode,wantResults
	
	Silent 1;PauseUpdate
	String dfSav= Set_IFDL_DataFolder()	// remember current data folder, set to WM_IFDL
	Variable/G dispMode= mode				// save mode for QuantizeDesignCoefficients(); see also ShowResults()

	String cwave= dfSav+"coefs"
	FixFilterDelay($cwave)

	Variable b=1
	String bandText=""
	if (exists("bandInfo")==1)
		bandText	= "\rBand\tStart Freq (Hz)\tEnd Freq (Hz)\r"		
		Variable i= 0;
		do
			bandText+=num2istr(b)+"\t"+num2str(bandInfo[i+1])+"\t\t"+num2str(bandInfo[i+2])
			if (bandInfo[i] == 1)
				bandText+= "\t(pass band)\r"
			else
				bandText+= "\t(reject band)\r"
			endif
			b += 1
			i+= 3
		while( i < numpnts(bandInfo) )
		bandText += "(end)\r"
	endif
	Note/K $cwave
	Note $cwave,designTypeName+bandText
	Duplicate/O $cwave preQuantCoefs	// in WM_IFDL, to try other quantizations later
	SetDataFolder dfSav
	if( wantResults )
		StdCoefResults(1)
	endif
End

Function FixFilterDelay(filter)
	Wave filter

	if( WaveExists(filter) == 0 )
		return 1
	endif
	NVAR ndlyCsl= root:Packages:WM_IFDL:ndlyCsl
	NVAR fs= root:Packages:WM_IFDL:fs
	Variable offset
	if( ndlyCsl == 2 )
		offset= 0
	else
		 offset= - trunc((numpnts(filter)-1)/2)/fs
	endif
	SetScale/P x,offset,1/fs,"s",filter
	return 0
End
	
// Lite version of StdCoefsTreatmentNoShowResults() that doesn't double-quantize coefs
Function FixCoefsDelay()
	WAVE filter= root:Packages:WM_IFDL:preQuantCoefs
	if( FixFilterDelay(filter) == 0 )
		Execute "StdCoefResults(1)	"	// transfer preQuantCoefs to coefs
	endif
End

Proc StdCoefResults(quantize)
	Variable quantize	 // if 0, don't quantize the coefficients
	
	Silent 1;PauseUpdate
	String dfSav= Set_IFDL_DataFolder()
	String cwave= dfSav+"coefs"
	Variable ncoefs=numpnts($cwave)
	if( quantize != 0 )
		ApplyQuant(preQuantCoefs,$cwave,quanType,quanBits)
	else
		Duplicate/O preQuantCoefs $cwave
		Print "Note: coefficients are not quantized"
	endif
	Duplicate/O $cwave coefs_tmp
	Variable ncoefsEven= 2 * trunc((ncoefs+1)/2)
	Variable fftln = max(2 * fftLen,ncoefsEven)
	Redimension/N=(fftln) coefs_tmp

	// eliminate linear phase due to delay
	Rotate -ncoefsEven/2+1, coefs_tmp

	FFT coefs_tmp
	Make/O/N=(numpnts(coefs_tmp)) coefsMag	, coefsPhase	// length fftln/2 + 1
	CopyScales coefs_tmp coefsMag, coefsPhase
	coefs_tmp= r2polar(coefs_tmp)
	coefsMag=real(coefs_tmp)
	coefsPhase= imag(coefs_tmp) * 180 / pi		// degrees
	KillWaves coefs_tmp
	
	Duplicate/O coefsMag coefsDbMag
	coefsDbMag=20*log(coefsDbMag)
	
	Duplicate/O $cwave coefsImpulse
	Redimension/N=(2*ncoefs) coefsImpulse
	Rotate ncoefs/2,coefsImpulse

	Duplicate/O coefsImpulse coefsStep
	SetScale/P x,0,1,coefsStep
	Integrate coefsStep
	CopyScales coefsImpulse coefsStep

	if(designFlags %& 0x1)				// passband blowup?
		Duplicate/O coefsDbMag dbmagPassDetails
		Duplicate/O coefsMag magPassDetails
		Duplicate/O coefsPhase phasePassDetails
		dbmagPassDetails=NaN
		magPassDetails=NaN
		phasePassDetails=NaN

		Variable i= 0;
		do
			if(bandInfo[i]==1)		// passband
				dbmagPassDetails(bandInfo[i+1],bandInfo[i+2])=coefsDbMag
				magPassDetails(bandInfo[i+1],bandInfo[i+2])=coefsMag
				phasePassDetails(bandInfo[i+1],bandInfo[i+2])=coefsPhase
			endif
			i+= 3
		while( i < numpnts(bandInfo) )
	endif

	if(designFlags %& 0x4)				// differentiator error curve?
		Duplicate/O coefsMag md_diffError
		md_diffError=NaN
		md_diffError(bandInfo[1],bandInfo[2])=coefsMag-bandInfo[0]*x/fs
	endif
	SetDataFolder dfSav
End


// By IFDL convention, an IFDL window Proc named "abc" creates
// a window whose name is "WMAbc". This helps prevent the user
// from over-writing the IFDL Proc when the user closes a window
// and clicks "Replace" in the Close Window dialog.
// 1 - Linear Magnitude
// 2 - DB Magnitude
// 4 - Impulse Response
// 8 - Step Response
// 16 - Pass Band Details
// 32 - Differentiator Error
// 64 - Arbitrary Design Results


Proc ShowResults(mode)
	Variable mode // bit field controls what is displayed

	Silent 1
	// CreateWMIFDLArbGraph() done first so that other
	// results can be displayed on top of it
	DoWindow WMIFDLArbGraph
	if( mode %& 0x40 ) // Arbitrary Graph
		if( V_Flag )
			DoWindow/F WMIFDLArbGraph
		else
			CreateWMIFDLArbGraph()
		endif
	else
		if( V_Flag )
			DoWindow/K WMIFDLArbGraph
		endif
	endif
	
	DoWindow WMDispLinMag
	if( mode %& 1 ) // Linear Magnitude requested
		if( V_Flag )
			DoWindow/F WMDispLinMag
		else
			dispLinMag()
		endif
	else
		if( V_Flag )
			DoWindow/K WMDispLinMag
		endif
	endif
	
	DoWindow WMDispDbMag
	if( mode %& 2 ) // DB Magnitude requested
		if( V_Flag )
			DoWindow/F WMDispDbMag
		else
			dispDbMag()
		endif
	else
		if( V_Flag )
			DoWindow/K WMDispDbMag
		endif
	endif

	DoWindow WMDispImpulse
	if( mode %& 4 ) // Impulse Response requested
		if( V_Flag )
			DoWindow/F WMDispImpulse
		else
			dispImpulse()
		endif
	else
		if( V_Flag )
			DoWindow/K WMDispImpulse
		endif
	endif

	DoWindow WMDispStep
	if( mode %& 8 ) // Step Response requested
		if( V_Flag )
			DoWindow/F WMDispStep
		else
			dispStep()
		endif
	else
		if( V_Flag )
			DoWindow/K WMDispStep
		endif
	endif

	DoWindow WMPassDetails
	if( mode %& 0x10 ) // Passband details requested
		if( V_Flag )
			DoWindow/F WMPassDetails
		else
			passDetails()
		endif
	else
		if( V_Flag )
			DoWindow/K WMPassDetails
		endif
	endif

	DoWindow WMDiffError
	if( mode %& 0x20 ) // Differentiator Error requested
		if( V_Flag )
			DoWindow/F WMDiffError
		else
			diffError()
		endif
	else
		if( V_Flag )
			DoWindow/K WMDiffError
		endif
	endif
End

// round to 1, 2, or 5 * 10eN, non-rigorously
Function NiceNumber(num)
	Variable num
	
	if( num == 0 )
		return 0
	endif
	Variable theSign= sign(num)
	num= abs(num)
	Variable lg= log(num)
	Variable decade= floor(lg)
	Variable frac = lg - decade
	Variable mant
	if( frac < log(1.5) )	// above 1.5, choose 2
		mant= 1
	else
		if( frac < log(4) )	// above 4, choose 5
			mant= 2
		else
			if( frac < log(8) )	// above 8, choose 10
				mant= 5
			else
				mant= 10
			endif
		endif
	endif
	num= theSign * mant * 10^decade
	return num
End

Function/S RemoveLastChar(str)
	String str
	
	Variable len=strlen(str)
	return str[0,len-2]
End

// Use: legend += DesignLegendPiece(traceName,traceText)
// use legend = RemoveLastChar(legend) to remove trailing \r
Function/S DesignLegendPiece(traceName,traceText)
	String traceName,traceText

	String path="root:Packages:WM_IFDL:"
	String legendPiece=""
	WAVE wv= $(path+traceName)
	CheckDisplayed wv
	if( V_Flag == 1 )
		legendPiece= DesignLegendSize()+"\\s("+traceName+") "+traceText+"\r"
	endif
	return legendPiece
End

Function/S DesignLegendSize()
	String legendSize=""	// auto size

	Variable fSize=str2num(StrVarOrDefault("root:Packages:WM_IFDL:annoFontSizeStr","auto"))	// NaN if "auto" or error
	if( fSize == limit(fSize,5,99) )
		sprintf legendSize, "\\Z%02d",fSize
	endif
	return legendSize
End

// Example: 	DesignLegend("WMKaiserLowPassDesign","klp","klpLegend")
Function DesignLegend(windowName,waveNamePrefix,legendName)
	String windowName
	String waveNamePrefix	// something like "mlp", creates legend entries for mlp_response, etc
	String legendName
	Variable isDB

	DoWindow $windowName
	if( V_Flag == 0 )
		return 0
	endif
	DoWindow/F $windowName
	
	// Desired Response
	String traceName= waveNamePrefix+"_response"
	String str= DesignLegendPiece(traceName,"Desired Response")

	// Desired DB response
	traceName= waveNamePrefix+"_responseDB"
	str += DesignLegendPiece(traceName,"Desired Response")

	// Actual response
	str += DesignLegendPiece("coefsMag","Actual Response")
	str += DesignLegendPiece("coefsDbMag","Actual Response")
	
	// pass band details
	str += DesignLegendPiece("magPassDetails","Pass Band Details")
	str += DesignLegendPiece("dbmagPassDetails","Pass Band Details")
	str += DesignLegendPiece("phasePassDetails", "\\K(65535,0,0)Phase in Pass Band\\K(0,0,0)") // to match the axis and trace	 

	traceName= waveNamePrefix+"_diffError"
	str += DesignLegendPiece(traceName,"Error from straight line")
	
	traceName= waveNamePrefix+"_notchFrequency"
	str += DesignLegendPiece(traceName,"Desired Notch Frequency")
	
	traceName= waveNamePrefix+"_passErrorPlus"
	str += DesignLegendPiece(traceName,"Maximum Pass Band Error")
	
	traceName= waveNamePrefix+"_transitionPlus"
	str += DesignLegendPiece(traceName,"Transition Region")
	
	traceName= waveNamePrefix+"_transitionRegions"
	str += DesignLegendPiece(traceName,"Transition Regions")

	TextBox/C/N=$legendName RemoveLastChar(str)
End

Proc dispStyle()
	Silent 1;PauseUpdate
	ModifyGraph lSize=0.5
	ModifyGraph minor=1
	ModifyGraph grid=2
	ModifyGraph standoff=1
	ModifyGraph margin(left)=60
	ModifyGraph margin(right)=22
End

Proc dispLinMag() : Graph
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMDispLinMag
	String dfSav= Set_IFDL_DataFolder()
	Variable x0=2*72/ScreenResolution, y0= 40*72/ScreenResolution
	Variable x1=252*72/ScreenResolution, y1= 200*72/ScreenResolution
	Display/K=1 /W=(x0,y0,x1,y1) coefsMag as "Magnitude Linear"
	SetDataFolder dfSav
	 dispStyle()
	Label left "response (linear)"
	DoWindow/C WMDispLinMag
EndMacro

Proc dispDbMag() : Graph
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMDispDbMag
	String dfSav= Set_IFDL_DataFolder()
	Variable x0=255*72/ScreenResolution, y0= 40*72/ScreenResolution
	Variable x1=503*72/ScreenResolution, y1= 200*72/ScreenResolution
	Display/K=1 /W=(x0,y0,x1,y1) coefsDbMag as "Magnitude in dB"
	SetDataFolder dfSav
	 dispStyle()
	ModifyGraph tick(left)=1
	ModifyGraph mirror(left)=1,mirror(bottom)=2
	Label left "response (dB)"
	DoWindow/C WMDispDbMag
EndMacro

Proc dispImpulse() : Graph
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMDispImpulse
	String dfSav= Set_IFDL_DataFolder()
	Variable x0=255*72/ScreenResolution, y0= 221*72/ScreenResolution
	Variable x1=503*72/ScreenResolution, y1= 379*72/ScreenResolution
	Display/K=1 /W=(x0,y0,x1,y1) coefsImpulse as "Impulse Response"
	SetDataFolder dfSav
	 dispStyle()
	ModifyGraph mode=8
	ModifyGraph marker=19
	Textbox/C/N=text0/S=3/A=MC/X=30.27/Y=34.86 "impulse response"
	DoWindow/C WMDispImpulse
EndMacro

Proc dispStep() : Graph
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMDispStep
	String dfSav= Set_IFDL_DataFolder()
	Variable x0=2*72/ScreenResolution, y0= 221*72/ScreenResolution
	Variable x1=252*72/ScreenResolution, y1= 379*72/ScreenResolution
	Display/K=1 /W=(x0,y0,x1,y1) coefsStep as "Step Response"
	SetDataFolder dfSav
	 dispStyle()
	Textbox/C/N=text0/S=3/A=MC/X=30.27/Y=34.86 "step response"
	DoWindow/C WMDispStep
EndMacro

Proc coefsTable() : Table
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMCoefsTable
	Edit/K=1/W=(309,41,501,430) coefs
	ModifyTable width(Point)=26
	ModifyTable format(coefs)=5,digits(coefs)=16,width(coefs)=138
	DoWindow/C WMCoefsTable
EndMacro

Proc passDetails() : Graph
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMPassDetails
	String dfSav= Set_IFDL_DataFolder()
	Variable x0=2*72/ScreenResolution, y0= 40*72/ScreenResolution
	Variable x1=252*72/ScreenResolution, y1= 200*72/ScreenResolution
	Display/K=1 /W=(x0,y0,x1,y1) dbmagPassDetails as "Passband Details"
	SetDataFolder dfSav
	ModifyGraph margin(left)=60
	dispStyle()
	ModifyGraph mirror(left)=1,mirror(bottom)=2
	ModifyGraph lowTrip(left)=0.001
	Label left "response (dB)"
	Textbox/C/N=text0/S=3/A=MC/X=30.27/Y=34.86 "pass band details"
	DoWindow/C WMPassDetails
EndMacro

Proc diffError() : Graph
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMDiffError
	String dfSav= Set_IFDL_DataFolder()
	Variable x0=255*72/ScreenResolution, y0= 40*72/ScreenResolution
	Variable x1=503*72/ScreenResolution, y1= 200*72/ScreenResolution
	Display/K=1/W=(x0,y0,x1,y1)  md_diffError as "Response Error"
	SetDataFolder dfSav
	ModifyGraph mirror(left)=1,mirror(bottom)=2
	dispStyle()
	Label left "response error"
	DoWindow/C WMDiffError
EndMacro

Proc IFDL_Layout0() : Layout
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMIFDLLayout0
	Layout/W=(3,42,389,475) WMCoefsTable(374,72,540,666)/O=2
	AppendToLayout WMDispImpulse(71,427,375,582)/O=1
	AppendToLayout WMDispStep(71,580,375,720)/O=1
	AppendToLayout WMIFDLArbGraph(71,153,375,429)/O=1
	Textbox/N=title/F=0/A=MT/X=-12.36/Y=5.79 "\\JCMcClellan-Parks; arbitrary response"
	Textbox/N=specs/F=0/A=MT/X=-14.36/Y=10.47 "\\Z08Sampling frequency:  1Hz\rNum terms: 41"
	AppendText "Quantization: Floating point, 53 bits"
	DoWindow/C WMIFDLLayout0
EndMacro

Proc IFDL_Layout1() : Layout
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMIFDLLayout1
	Layout/W=(2,41,368,486) WMCoefsTable(373,72,539,666)/O=2
	AppendToLayout WMDispStep(72,590,374,720)/O=1
	AppendToLayout WMDispImpulse(72,452,374,596)/O=1
	AppendToLayout WMDispDbMag(72,287,374,454)/O=1
	AppendToLayout WMPassDetails(72,165,374,289)/O=1
	Textbox/N=title/F=0/A=MT/X=-12.36/Y=6.06 "\\JCMcClellan-Parks; two band"
	Textbox/N=specs/F=0/A=MT/X=-12.73/Y=9.92 "\\Z08Sampling frequency:  44100Hz\rNum terms: 29"
	AppendText "Quantization: Floating point, 53 bits\rBand 1: passband[0,8820 Hz], peak error +0.465,-0.554dB"
	AppendText "Band 2: stopband[11025,22050 Hz], min reject -31.6dB"
	DoWindow/C WMIFDLLayout1
EndMacro


Proc IFDL_Layout2() : Layout
	PauseUpdate; Silent 1		// building window...
	DoWindow/K WMIFDLLayout2
	Layout /W=(5,41,408,477) WMCoefsTable(375,72,541,722)/O=2
	AppendToLayout WMDispImpulse(72,458,376,598)/O=1
	AppendToLayout WMDispStep(72,597,376,721)/O=1
	AppendToLayout WMDiffError(72,331,376,459)/O=1
	AppendToLayout WMDispLinMag(72,188,376,332)/O=1
	Textbox/N=title/F=0/A=MT/X=-14.55/Y=7.02 "\\JCMcClellan-Parks; differentiator"
	Textbox/N=specs/F=0/A=MT/X=-15.64/Y=12.26 "\\Z08Sampling frequency:  1Hz\rNum terms: 10\rQuantization: Floating point, 53 bits"
	AppendText "Desired slope: 1"
	DoWindow/C WMIFDLLayout2
EndMacro


Proc SineSweepTestSignal(fstart,fend,linLog,taper,tstw)
	Variable fstart=NumVarOrDefault("root:Packages:WM_IFDL:sst_fs",0.01) * NumVarOrDefault("root:Packages:WM_IFDL:fs",1)
	Variable fend=NumVarOrDefault("root:Packages:WM_IFDL:sst_fe",0.5) * NumVarOrDefault("root:Packages:WM_IFDL:fs",1)
	Variable linLog=NumVarOrDefault("root:Packages:WM_IFDL:sst_lnlg",1)
	Variable taper=NumVarOrDefault("root:Packages:WM_IFDL:sst_tpr",5) /  NumVarOrDefault("root:Packages:WM_IFDL:fs",1)
	String tstw=StrVarOrDefault("root:Packages:WM_IFDL:sst_dest","sineSweep")
	Prompt fstart "start freq"+StrVarOrDefault("root:Packages:WM_IFDL:fsStr","")
	Prompt fend "end frequency"
	Prompt linLog "freq sweep mode",popup,";linear;log"
	Prompt taper "rise/fall time (sec)"
	Prompt tstw "output wave"
	
	PauseUpdate;Silent 1	// Sine Sweep Test Signal...
	tstw= CleanupName(tstw,1)
	GuaranteeIFDLInitialized()
	String dfSav= Set_IFDL_DataFolder()
	Variable/G sst_fs=fstart/fs
	Variable/G sst_fe=fend/fs
	Variable/G sst_lnlg=linLog
	Variable/G sst_tpr=taper*fs
	String/G sst_dest=tstw
	SetDataFolder dfSav
	if( ChkFreq(fstart/root:Packages:WM_IFDL:fs,fend/root:Packages:WM_IFDL:fs,0,0) )
		abort
	endif
	Variable n= root:Packages:WM_IFDL:fftLen*8
	Make/O/N=(n) $tstw	// new wave in user's data folder

	SetScale/P x,0,1/root:Packages:WM_IFDL:fs,"s",$tstw
	Variable xs= leftx($tstw), xe = rightx($tstw)
	if(linLog==1)
		$tstw = 2*pi*(fstart+(fend-fstart)*(x-xs)/(xe-xs)) // linear is best for viewing frequency response
	else
		$tstw =2*pi*(fstart+(fend-fstart)*(alog((x-xs)/(xe-xs))-1)/9)	// log is best for viewing time response
	endif
	Integrate $tstw
	$tstw=sin($tstw)*REdge(x,leftx($tstw)+taper,taper,1)*REdge(x,rightx($tstw)-taper,taper,0)

	ShowWaveInGraphWithTitle($tstw,"WMSineSweep","Sine Sweep Test Signal")
	DoWindow/F WMSineSweep
End

Function ShowWaveInGraphWithTitle(wv,graphName,title)
	Wave wv
	String graphName,title
	
	Variable appended= 0
	DoWindow $graphName
	if( V_Flag == 0 )	// window doesn't exist
		Display/K=1
		DoWindow/C $graphName
		AutopositionWindow/E $graphName
		if( strlen(title ) )
			DoWindow/T $graphName, title
		endif
		Legend/C/N=ifdlLegend
	endif
	CheckDisplayed/W=$graphName wv
	if( V_Flag == 0 )	// not displayed
		DoWindow/F $graphName
		AppendToGraph wv
		Execute "Distinguish()"
		appended= 1
	endif
	return appended
End

Proc Distinguish() : GraphStyle
	PauseUpdate; Silent 1		// modifying window...
	ModifyGraph/Z lStyle[0]=0,lStyle[1]=1,lStyle[2]=2,lStyle[3]=3,lStyle[4]=4,lStyle[5]=3
	ModifyGraph/Z lStyle[6]=2
	ModifyGraph/Z marker[0]=19,marker[1]=18,marker[2]=17,marker[3]=16
	ModifyGraph/Z marker[4]=9,marker[5]=8,marker[6]=7
	ModifyGraph/Z rgb[0]=(65535,0,0),rgb[1]=(0,0,65535),rgb[2]=(1534,24804,2111),rgb[3]=(0,0,0)
	ModifyGraph/Z rgb[4]=(0,0,65535),rgb[5]=(2188,36662,3822),rgb[6]=(36662,2188,3822)
	ModifyGraph/Z lSize=.5,gaps=0
	SplitLeftAndRightAxes()
EndMacro

// TraceAxis() returns a string containing the X or Y axis of the named trace.
//
Function/S TraceAxis(graph,trace,instance,wantY)
	String graph,trace
	Variable instance		// 0 is first instance, 1 is second, ...
	Variable wantY
	
	String info=TraceInfo(graph,trace,instance)
	Variable start,theEnd
	if( wantY )
		start= strsearch(info,"YAXIS:",0)
	else
		start= strsearch(info,"XAXIS:",0)
	endif
	if( start < 0 )
		return ""
	endif
	start += 6
	theEnd= strsearch(info,";",start)
	if( (start >= 0) %& (theEnd >= start) )
		return info[start,theEnd-1]
	endif
	return ""
End

Function SplitLeftAndRightAxes()
	String list= AxisList("")
	Variable haveBoth = (strsearch(list,"left;",0) >= 0) %& (strsearch(list,"right;",0) >= 0)
	if( haveBoth )
		String info= AxisInfo("","left")
		Variable st= GetNumFromModifyStr(info,"axisEnab","{",0)
		Variable en= GetNumFromModifyStr(info,"axisEnab","{",1)
		if( (st <= 0) %& (en >= 1) )
			ModifyGraph/Z axisEnab(left)={0,0.45}, axisEnab(right)={0.55,1}
		endif
	else
		ModifyGraph/Z axisEnab(left)={0,1}, axisEnab(right)={0,1}
	endif
End


// ********** Controls Utilities

Function/S ControlTitle(win,ctrl)
	String win,ctrl
	
	String typeList="Button ;CheckBox ;PopupMenu ;ValDisplay ;SetVariable ;Chart ;"
	ControlInfo/W=$win $ctrl
	if( V_Flag < 0 )
		return ""		// no control by that name
	endif
	String controlType= GetStrFromList(typeList, abs(V_Flag)-1, ";")
	String rec= WinRecreation(win,0)
	String title="",line
	Variable start,theEnd=0, st,en
	do
		start= strsearch(rec,"Button "+ctrl+",",theEnd)
	
		if( start < 0 )
			break
		endif
		theEnd= strsearch(rec,"\r",start)
		line= rec[start,theEnd-1]
		st= strsearch(line,"title=\"",0)
		if( st >= 0 )
			st += 7	// skip title="
			en= FindUnQuotedChar(line,"\"",st)
			if( en >= 0 )
				title= line[st,en-1]
				break;
			endif
		endif
		
	while( 1 )
	
	return title
End

// Finds a non-quoted character in the string.
// returns the index, just like strsearch
// Note: chr is case-insensitive
Function FindUnQuotedChar(str,chr,start)
	String str,chr
	Variable start
	
	Variable index= -1	// not found
	Variable st= start
	Variable backslash= char2num("\\")
	
	do
		start= strsearch(str,chr,start)
		if( start < 0 )
			break
		endif
		// found the char, check if it is quoted
		if ( (start > st) %& (char2num(str[start-1]) == backslash) )
			start += 1	// quoted chr, skip it
		else
			index= start
			break
		endif
	while (1)
	return index
End

// ********** Window Coordinates Save/Restore Utilities


Function WindowCoordinatesRow(coords,windowName)
	Wave/T coords
	String windowName
	
	Variable rows = DimSize(coords,0)
	Variable row= -1
	if( rows > 0 )
		do
			row += 1
			if( CmpStr(windowName,coords[row][0]) == 0 )
				return row
			endif
		while( row < rows-1 )
		row= -1	// not found
	endif
	return row
end

Function/S WindowNameOrTarget(windowName)
	String windowName
	
	if( strlen(windowName) == 0 )
		windowName= WinName(0,255)
	endif
	return windowName
End

Function WindowCoordinatesHook(infoStr)
	String infoStr
	Variable statusCode= 0
	String event= StrByKey("EVENT",infoStr)
	if( CmpStr(event,"kill") == 0 )
		String windowName= StrByKey("WINDOW",infoStr)
		WindowCoordinatesSave(windowName)
		statusCode= 1
	endif
	return statusCode
End

// window coordinates are saved in a 5-column text wave
// as windowname,num2istr(left),num2istr(top),num2istr(right),num2istr(bottom)
Function WindowCoordinatesSave(windowName)
	String windowName
	
	windowName= WindowNameOrTarget(windowName)
	DoWindow $windowName
	if( V_Flag == 0 )
		return 0
	endif
	Variable row
	String dfSav= Set_IFDL_DataFolder()
	if( exists("W_windowCoordinates") != 1 )
		Make/O/T/N=(0,5) W_windowCoordinates
		WAVE/T coords= root:Packages:WM_IFDL:W_windowCoordinates
		row= -1	// add new row
	else
		// search for matching row
		WAVE/T coords= root:Packages:WM_IFDL:W_windowCoordinates
		row= WindowCoordinatesRow(coords,windowName)
	endif
	if( row == -1 )
		InsertPoints/M=0 0,1,coords
		row= 0
	endif
	GetWindow $windowName, wsize		// always returns points
	if( WinType(windowName) == 7 )	// but NewPanel/W uses pixels
		V_left *= ScreenResolution/72
		V_top *= ScreenResolution/72
		V_right *= ScreenResolution/72
		V_bottom *= ScreenResolution/72
	endif
	coords[row][0]=windowName
	coords[row][1]=num2str(V_left)
	coords[row][2]=num2str(V_top)
	coords[row][3]=num2str(V_right)
	coords[row][4]=num2str(V_bottom)
	SetDataFolder dfSav
	return 1
End

// returns "" or the coordinates separated by commas
// puts window coordinates (in points) in globals coords_left, etc.
Function/S WindowCoordinatesGet(windowName)
	String windowName

	windowName= WindowNameOrTarget(windowName)
	WAVE/T coords= root:Packages:WM_IFDL:W_windowCoordinates
	if( WaveExists(coords) == 0 )
		return ""
	endif
	Variable row= WindowCoordinatesRow(coords,windowName)
	if( row < 0 )
		return ""
	endif
	String left= coords[row][1]
	String top= coords[row][2]
	String right= coords[row][3]
	String bottom= coords[row][4]

	String coordinates=left+", "+top+", "+right+", "+bottom
	return coordinates
End

// note, the window must exist before calling this
// returns the command that moved the window
Function WindowCoordinatesRestore(windowName)
	String windowName

	windowName= WindowNameOrTarget(windowName)
	DoWindow $windowName
	if( V_Flag == 0 )
		return 0
	endif
	String coordinates= WindowCoordinatesGet(windowName)
	if( strlen(coordinates) > 0 )
		String cmd="DoWindow/F "+windowName+"; MoveWindow "+coordinates
		Execute cmd
	endif
	return 1
End

// WindowCoordinatesSprintf
// %s in fmt is replaced with left,top,right,bottom
//
// Example:
// String fmt="Display/W=(%s) as \"the title\""
// Execute WindowCoordinatesSprintf("eventualWindowName",fmt,x0,y0,x1,y1)
//
Function/S WindowCoordinatesSprintf(windowName,fmt,defLeft,defTop,defRight,defBottom)
	String windowName,fmt
	Variable defLeft,defTop,defRight,defBottom
	
	String coordinates= WindowCoordinatesGet(windowName)
	if( strlen(coordinates) == 0 )
		sprintf coordinates, "%g, %g, %g, %g",defLeft,defTop,defRight,defBottom
	endif
	String result
	Sprintf result, fmt, coordinates
	return result
end

Function DesignGraph(graphName,title)
	String graphName,title
	
	DoWindow $graphName
	Variable existed= V_Flag
	if( existed == 0 )
		KillAllFilterDesignWindows()
		Variable x0=6*72/ScreenResolution, y0= 42*72/ScreenResolution
		Variable x1=490*72/ScreenResolution, y1= 327*72/ScreenResolution
		String fmt="Display/K=1/W=(%s) as \"" + title + "\""
		Execute WindowCoordinatesSprintf(graphName,fmt,x0,y0,x1,y1)
		DoWindow/C $graphName
	endif
	return existed
End

// ********** Quick Drag Utilities

// Quick drag mode (quickdrag=1) is a special purpose mode provided to allow
// the creation of cross hair cursors using a package of Igor procedures.
// (See the Cross Hair Demo example experiment.) Normally you would have
// to click and hold on a trace for one second before entering drag mode.
// When quickdrag is in effect, there is no delay.
// 
// If a trace is in quickdrag mode it should also be set to live mode.
// This combination allows you to click on a trace and immediately drag
// it to a new x and y offset.
// 
// In addition to quick drag mode, the cross hair package relies on Igor
// to store information about the drag in a string variable if certain
// conditions are in effect.
// 
// The string variable name (that you have to create) is S_TraceOffsetInfo.
// It must reside in a data folder that has the same name as the graph
// (not title!!) which in turn must reside in root:WinGlobals:.
// 
// If these conditions are met, then after a trace is dragged, Igor will
// store information in the string using the following key-value format:
// GRAPH:<name of graph>;XOFFSET:<x offset value>;YOFFSET:<y offset value>;TNAME:<trace name>;

// Example:
// ...
// String pathToInfoStr= QuickDragForWindowSetup("Graph0","MyFunction")
// End
// Function MyFunction(str)
//		String str
// ...
// end
// returns path to trace info string

Function/S QuickDragForWindowSetup(windowName,functionName,killDrag)
	String windowName,functionName
	Variable killDrag
	
	String traceInfoPath= "root:WinGlobals:"+windowName+":S_TraceOffsetInfo"

	if( killDrag )
		KillStrings/Z root:WinGlobals:$(windowName):S_TraceOffsetInfo
		KillVariables/Z root:WinGlobals:$(windowName):updatesOnQuickDrag
	else
		NewDataFolder/O root:WinGlobals
		NewDataFolder/O root:WinGlobals:$(windowName)
		String/G root:WinGlobals:$(windowName):S_TraceOffsetInfo= ""	
		if( exists(functionName) == 6 )	// if function name
			Variable/G root:WinGlobals:$(windowName):updatesOnQuickDrag
			SetFormula root:WinGlobals:$(windowName):updatesOnQuickDrag,functionName+"("+traceInfoPath+")"
		endif
	endif
	return traceInfoPath
End

// update desired response wave to reflect current filter response (from coefsMag or coefsDbMag, if either is displayed in the graph)
// unless the desired response is lowpass
// and the last filter design was highpass, and vice versa.

Function DesiredFromActualResponse(wDesiredResponse,graphName,b1e,b2s,b2eOrZero,b3sOrZero,firstBandIsPass,epsilon)
	Wave wDesiredResponse	// this wave must exist
	String graphName
	Variable b1e,b2s,b2eOrZero,b3sOrZero,firstBandIsPass,epsilon

	NVAR fs= root:Packages:WM_IFDL:fs
	Variable b1Response,b2Response,b3Response
	if( firstBandIsPass )
		b1Response= 1		// create desired response in linear form, then convert to db later
		b2Response= 1e-6
	else
		b1Response= 1e-6
		b2Response= 1
	endif
	b3Response= b1Response

	Variable threeBands= (b2eOrZero > 0)

	// If either response trace is on the graph, coefsMag is the result of 
	// a computation of the current design window, so we use it
	WAVE wCurrentResponseOrNull= $""
	DoWindow $graphName
	if( V_Flag == 1 )
		CheckDisplayed/W=$graphName root:Packages:WM_IFDL:coefsMag,root:Packages:WM_IFDL:coefsDbMag
		if( V_Flag != 0 )
			WAVE wCurrentResponseOrNull= root:Packages:WM_IFDL:coefsMag	// create desired response in linear form, then convert to dB later
		endif
	endif

	// If the current response wave exists,
	// conform the desired response levels to maximum of the reject band(s)
	Variable useCurrentResponse= WaveExists(wCurrentResponseOrNull)
	if( useCurrentResponse )
		Variable b2e= b2eOrZero
		Variable b3s= b3sOrZero
		if( !threeBands )
			b2e= fs/2+epsilon
			b3s= b2e
		endif
		Variable b1eFudged= limit(b1e-epsilon,epsilon,b1e)
		Variable b2sFudged= limit(b2s+epsilon,b2s,b2e-epsilon)
		Variable b2eFudged= limit(b2e-epsilon,b2s+epsilon,b3s)
		Variable b1avg= mean(wCurrentResponseOrNull,0,b1eFudged)
		Variable b2avg= mean(wCurrentResponseOrNull,b2sFudged,b2eFudged)
		Variable currentHasSameShape= firstBandIsPass %^ (b1avg < b2avg )
		if( currentHasSameShape )
			if( firstBandIsPass )
				b2Response= MaximumInBand(wCurrentResponseOrNull,b2sFudged,b2eFudged)
			else
				b1Response= MaximumInBand(wCurrentResponseOrNull,0,b1eFudged)
				if(threeBands)
					Variable b3sFudged= limit(b3s+epsilon,b3s,fs/2)
					b3Response= MaximumInBand(wCurrentResponseOrNull,b3sFudged,fs/2)
				endif
			endif
		endif
	endif
	if( threeBands )
		wDesiredResponse= {b1Response,b1Response,NaN,b2Response,b2Response,NaN,b3Response,b3Response}
	else
		wDesiredResponse= {b1Response,b1Response,NaN,b2Response,b2Response}
	endif
	if( dBIsChecked(graphName) )
		wDesiredResponse= 20*log(wDesiredResponse)
	endif
	if( firstBandIsPass )
		return b2Response
	else
		return b1Response
	endif
End

Function/S responseName(wantDB,wantPath)
	Variable wantDB,wantPath
	
	String traceName=""
	if( wantPath )
		traceName+="root:Packages:WM_IFDL:"
	endif
	if( wantDB )
		traceName+= "coefsDbMag"
	else
		traceName+="coefsMag"
	endif
	return traceName
End

Function StringEndsWith(suffix,str)
	String suffix,str
	Variable slen= strlen(str)
	Variable sfxlen= strlen(suffix)
	if( (sfxlen > 0) %& (CmpStr(suffix,str[slen-sfxlen,inf]) == 0) )
		return 1
	endif
	return 0
End

Function IsWindowsPlatform()
	return strsearch(IgorInfo(2),"Windows",0) >= 0
End
